import { supabase } from '@/config/supabase';
import {
  Referral,
  ReferralStatus,
  ReferralSource,
  RegisterReferralRequest,
  AdminReviewReferralRequest
} from './types';

export const referralsService = {
  // Register referral (during signup)
  async registerReferral(referred_user_id: string, req: RegisterReferralRequest) {
    // 1. Find referrer by code
    const { data: referrer } = await supabase.from('users').select('id').eq('referral_code', req.referrer_code).single();
    if (!referrer) return { error: { message: 'Invalid referral code.' } };
    if (referrer.id === referred_user_id) return { error: { message: 'Self-referral is not allowed.' } };
    // 2. Prevent duplicate referrals
    const { data: existing } = await supabase.from('referrals').select('id').eq('referred_user_id', referred_user_id).single();
    if (existing) return { error: { message: 'Referral already registered.' } };
    // 3. Insert referral
    const { data, error } = await supabase.from('referrals').insert({
      referrer_id: referrer.id,
      referred_user_id,
      source: 'signup',
      status: 'pending',
      reward: 0
    }).select().single();
    return { data, error };
  },

  // List my referrals
  async listReferrals(user_id: string) {
    const { data, error } = await supabase.from('referrals').select('*,referred_user:referred_user_id(full_name,email,created_at)').eq('referrer_id', user_id);
    return { data, error };
  },

  // Referral summary
  async getSummary(user_id: string) {
    const { data: referrals } = await supabase.from('referrals').select('*').eq('referrer_id', user_id);
    const total_referred = referrals?.length || 0;
    const qualified = referrals?.filter(r => r.status === 'qualified').length || 0;
    const pending = referrals?.filter(r => r.status === 'pending').length || 0;
    const total_earned = referrals?.filter(r => r.status === 'paid').reduce((sum, r) => sum + (r.reward || 0), 0) || 0;
    const paid_out = total_earned; // For now, all paid is paid out
    return {
      data: {
        total_referred,
        qualified,
        pending,
        total_earned,
        paid_out
      }
    };
  },

  // Admin: approve/reject referral
  async adminReviewReferral(id: string, req: AdminReviewReferralRequest, admin_id: string) {
    const { data: referral, error: fetchError } = await supabase.from('referrals').select('*').eq('id', id).single();
    if (fetchError) return { error: fetchError };
    // Only allow if status is qualified
    if (referral.status !== 'qualified') return { error: { message: 'Only qualified referrals can be paid or rejected.' } };
    const { data, error } = await supabase.from('referrals').update({ status: req.status }).eq('id', id).select().single();
    if (error) return { error };
    // Log admin action (if audit table exists)
    await supabase.from('admin_actions').insert({
      admin_id,
      action: `${req.status}_referral`,
      target_referral_id: id
    });
    return { data };
  }
}; 